/* Symbol table support
   Copyright (C) 1998 James Bowman

This file is part of gpasm.

gpasm is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

gpasm is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with gpasm; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include "stdhdr.h"

#include "gpasm.h"
#include "symbol.h"
#include "lst.h"
#include "cod.h"

#define HASH_SIZE 173  /* Too small and we get collisions.  Too big
			* and we use up memory and run slow.. */

struct symbol_table {
  int count;
  int case_insensitive;
  int (*compare)(const char *__s1, const char *__s2);
  struct symbol *hash_table[HASH_SIZE];
  struct symbol_table *prev;
};

static files;

/************************************************************************/

/* Base the hash func on the 1st, 2nd, 3rd and last characters of the
   string, and its length. */

static int hashfunc(struct symbol_table *t, char *s)
{
  union {
    char b[4];
    unsigned long ul;
  } change;
  int len;

  change.ul = 0;

  len = strlen(s);
  change.b[0] = s[0];
  if (len > 1) {
    change.b[1] = s[1];
    change.b[3] = s[len - 1];
    if (len > 2)
      change.b[2] = s[2];
  }

  if (t->case_insensitive) {
    change.ul &= 0x1f1f1f1f;
  }
  change.ul += (len << 3);

  return (change.ul % HASH_SIZE);
}

struct symbol_table *push_symbol_table(struct symbol_table * table,
				       int case_insensitive)
{
  struct symbol_table *new = calloc(sizeof(*new), 1);

  new->case_insensitive = case_insensitive;
  if (case_insensitive)
    new->compare = strcasecmp;
  else
    new->compare = strcmp;
  new->prev = table;
  return new;
}

struct symbol_table *pop_symbol_table(struct symbol_table *table)
{
  struct symbol_table *prev;

  prev = table->prev;

  return prev;
}

struct symbol *add_symbol(struct symbol_table *table, char *name)
{
  struct symbol *r;
  int index = hashfunc(table, name);

  table->count++;
  r = table->hash_table[index];
  while (r && ((*table->compare)(name, r->name) != 0))
    r = r->next;

  if (!r) {     /* No match */
    r = malloc(sizeof(*r));
    r->name = strdup(name);
    r->next = table->hash_table[index];
    r->annotation = NULL;
    table->hash_table[index] = r;
  }

  return r;
}

struct symbol *get_symbol(struct symbol_table *table, char *name)
{
  struct symbol *r = NULL;

  if (table != NULL) {
    int index = hashfunc(table, name);

    r = table->hash_table[index];
    while (r && ((*table->compare)(name, r->name) != 0))
      r = r->next;

    /* If r is still NULL, we didn't match.  Try the prev table on the stack */
    if (r == NULL)
      r = get_symbol(table->prev, name);
  }

  return r;
}

void annotate_symbol(struct symbol *s, void *a)
{
  s->annotation = a;
}

/************************************************************************/

char *get_symbol_name(struct symbol *s)
{
  return s->name;
}

void *get_symbol_annotation(struct symbol *s)
{
  return s->annotation;
}

/************************************************************************/

static int symbol_compare(const void *p0,
			  const void *p1)
{
  struct symbol
    *s0 = *(struct symbol **)p0,
    *s1 = *(struct symbol **)p1;
  return strcmp(s0->name, s1->name);
}

/* append the symbol table to the .lst file */

void dump_symbol_table(struct symbol_table *table)
{
  int i;
  const char *symbol_format = "%-32s  %08X";
  struct symbol **lst, **ps, *s;
  char buf[BUFSIZ];

  lst_line("");
  lst_line("SYMBOL TABLE");
  sprintf(buf, "%-32s  %-8s", "  LABEL", "  VALUE");
  lst_line(buf);
  lst_line("");

  ps = lst = malloc(table->count * sizeof(lst[0]));

  for (i = 0; i < HASH_SIZE; i++)
    for (s = table->hash_table[i]; s; s = s->next) 
      *ps++ = s;

  assert(ps == &lst[table->count]);

  qsort(lst, table->count, sizeof(lst[0]), symbol_compare);

  for (i = 0; i < table->count; i++) {
      struct variable *var;

      var = get_symbol_annotation(lst[i]);
      sprintf(buf,
	      symbol_format, 
	      get_symbol_name(lst[i]),
	      var ? var->value : 0);
      lst_line(buf);
    }
  cod_write_symbols(lst,table->count);
}

/* tsd: The next three routines: sym_init,sym_close,& sym_emit
 * currently (0.0.8) only support .cod symbol files.
 */

/* sym_init: Initialize the .sym file */

void sym_init()
{
  char *pc;
  int i;

  //  state.codfile = suppress;

  switch (state.codfile) {
  case suppress:
    state.cod.f = NULL;
    state.cod.enabled = 0;
    break;
  case normal:
    strcpy(state.codfilename, state.basefilename);
    strcat(state.codfilename, ".cod");
  case named:
    /* Don't need to do anything - name is already set up */
    state.cod.f = fopen(state.codfilename, "w");
    if (state.cod.f == NULL) {
      perror(state.codfilename);
      exit(1);
    }
    add_file(ft_cod,state.codfilename);
    state.cod.enabled = 1;
  }

  init_cod();

}

void sym_close()
{

  cod_close_file();

}

void sym_emit(unsigned int value)
{

  cod_emit_opcode(state.org,value);

}
