%{
/* Parser for gpasm
   Copyright (C) 1998 James Bowman

This file is part of gpasm.

gpasm is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

gpasm is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with gpasm; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include "stdhdr.h"

#include "gpasm.h"
#include "symbol.h"
#include "gperror.h"
#include "opcode.h"

void yyerror(char *message)
{
  gperror(103, message);
}

int yylex(void);

/************************************************************************/

/* Some simple functions for building parse trees */

static struct pnode *mk_pnode(enum pnode_tag tag)
{
  struct pnode *new = malloc(sizeof(*new));
  new->tag = tag;
  return new;
}

struct pnode *mk_constant(int value)
{
  struct pnode *new = mk_pnode(constant);
  new->value.constant = value;
  return new;
}

static struct pnode *mk_symbol(char *value)
{
  struct pnode *new = mk_pnode(symbol);
  new->value.symbol = value;
  return new;
}

static struct pnode *mk_string(char *value)
{
  struct pnode *new = mk_pnode(string);
  new->value.string = value;
  return new;
}

struct pnode *mk_list(struct pnode *head, struct pnode *tail)
{
  struct pnode *new = mk_pnode(list);
  new->value.list.head = head;
  new->value.list.tail = tail;
  return new;
}

static struct pnode *mk_2op(int op, struct pnode *p0, struct pnode *p1)
{
  struct pnode *new = mk_pnode(binop);
  new->value.binop.op = op;
  new->value.binop.p0 = p0;
  new->value.binop.p1 = p1;
  return new;
}

static struct pnode *mk_1op(int op, struct pnode *p0)
{
  struct pnode *new = mk_pnode(unop);
  new->value.unop.op = op;
  new->value.unop.p0 = p0;
  return new;
}

/************************************************************************/

%}

/* Bison declarations */

%union {
  gpasmVal i;
  char *s;
  struct pnode *p;
}

%token <s> LABEL
%token <s> IDENTIFIER
%token <s> CBLOCK, ENDC
%token <i> NUMBER
%token <s> STRING
%token <i> HIGH
%token <i> LOW
%token <i> LSH, RSH
%token <i> GREATER_EQUAL, LESS_EQUAL, EQUAL, NOT_EQUAL, '<', '>'
%token <i> '&', '|', '^'
%token <i> LOGICAL_AND, LOGICAL_OR
%token <i> '=', ASSIGN_PLUS, ASSIGN_MINUS, ASSIGN_MULTIPLY, ASSIGN_DIVIDE
%token <i> ASSIGN_MODULUS, ASSIGN_LSH, ASSIGN_RSH
%token <i> ASSIGN_AND, ASSIGN_OR, ASSIGN_XOR, INCREMENT, DECREMENT
%token <i> TBL_NO_CHANGE, TBL_POST_INC, TBL_POST_DEC, TBL_PRE_INC

%type <i> '+', '-', '*', '/', '%', '!', '~'
%type <s> line
%type <i> statement
%type <p> parameter_list
%type <p> expr, e0, e1, e2, e3, e4, e5, e6, e7, e8
%type <i> e1op, e2op, e3op, e4op, e5op, e6op, e7op, e8op

%%
/* Grammar rules */

program:
	/* can be nothing */
	|
	program { state.lst.line.was_org = state.org; state.lst.line.linetype = none; } line
	| program error '\n'
	{ 
	  next_line(0);
	}
	;

line:
	LABEL '=' expr '\n'
	{
	  gpasmVal value;

	  if (asm_enabled()) {
	    value = do_or_append_insn("set", mk_list($3, NULL));
	    if (!state.mac_prev) {
	      set_global($1, value, TEMPORARY, gvt_constant);
	    } else {
	      state.mac_body->label = $1;
	    }
	  }
	  next_line(value);
	}
	|
	LABEL statement
	{
	  if (state.lst.line.linetype == none)
	    state.lst.line.linetype = insn;
	  
	  if (asm_enabled()) {
	    if (state.mac_head) {
	      /* This is a macro definition.  Set it up */
	      struct symbol *mac;
	      struct macro_head *h = NULL;

	      mac = get_symbol(state.stMacros, $1);
	      if (mac)
		h = get_symbol_annotation(mac);

	      /* It's not an error if macro was defined on pass 1 and
		 we're in pass 2. */
	      if (h &&
		  !((h->pass == 1) && (state.pass == 2))) {
		char complaint[BUFSIZ];

		sprintf(complaint,
			"Attempt to redefine macro - first definition at line %d",
			h->line_number);
		gperror(118, complaint);
	      } else {
		if (!mac)
		  mac = add_symbol(state.stMacros, $1);
		annotate_symbol(mac, state.mac_head);
		h = state.mac_head;
		h->line_number = state.src->line_number;
	      }
	      h->pass = state.pass;

	      state.mac_head = NULL;
	    } else if (!state.mac_prev) {
	      /* Outside a macro, just define the label. */
	      switch (state.lst.line.linetype) {
	      case set:
		set_global($1, $2, TEMPORARY, gvt_constant);
		break;
	      case equ:
		set_global($1, $2, PERMANENT, gvt_constant);
		break;
	      case insn:
		set_global($1, $2 << _16bit_core, PERMANENT, gvt_address);
		break;
	      default:
	      }
	    } else {
	      /* Inside a macro; add the label to the current line */
	      if (state.mac_body)
		state.mac_body->label = $1;
	    }
	  }
	    
	  /*printf("symbol  %s %s at line %d\n", $1, state.src->line_number);*/
	  next_line($2);
	}
	|
	statement
	{
	  if (state.mac_head) {
	    /* This is a macro definition, but the label was missing */
	    state.mac_head = NULL;
	    gperror(117, "Missing macro label");
	  } else {
	    next_line(0);
	  }
	}
	;

statement:
	'\n'
	{
	  if (!state.mac_prev) {
	    $$ = state.org;
	  } else {
	    macro_append();
	  }
	}
	|
	IDENTIFIER '\n'
	{
	  $$ = do_or_append_insn($1, NULL);
	}
	|
	IDENTIFIER parameter_list '\n'
	{
	  $$ = do_or_append_insn($1, $2);
	}
	|
	CBLOCK expr '\n' { begin_cblock($2); next_line(0); }
        const_block
	ENDC '\n'
	{
	  $$ = 0;
	}
        |
        CBLOCK '\n' { next_line(0); }
        const_block
        ENDC '\n'
        {
           $$ = 0;
        }
	|
	CBLOCK error ENDC '\n'
	{
	  $$ = 0;
  	}
	;

const_block:
	|
	const_block const_line
	{
	  next_line(0);
	}
	;

const_line:
	'\n'
	|
	const_def_list '\n'
	|
	LABEL '\n'
	{
	  cblock_expr($1);
	}
	|
	LABEL expr '\n'
	{
	  cblock_expr_incr($1, $2);
	}

	;

const_def_list:
	const_def
	|
	const_def_list ',' const_def
	;

const_def:
	IDENTIFIER
	{
	  cblock_expr($1);
	}
	|
	IDENTIFIER ':' expr
	{
	  cblock_expr_incr($1, $3);
	}
	;

parameter_list:
	expr
	{
	  $$ = mk_list($1, NULL);
	}
	|
	expr ',' parameter_list
	{
	  $$ = mk_list($1, $3);
	}
	;

expr:
	e8
	|
	STRING
        {
	  /* Cook up the string, replacing \X with its code. */
	  char *ps, *pd;

	  ps = pd = $1;
	  while (*ps) {
	    if (*ps != '\\') {
	      *pd++ = *ps++;
	    } else {
	      ps++; /* Go past the escape */
	      switch (*ps) {
	      case 'n':
		*pd++ = '\n';
		break;
	      }
	      ps++;
	    }
	  }
	  *pd = 0;
	  $$ = mk_string($1);
        }
	;

e8:
	e7
	|
	e8 e8op e7
	{
	  $$ = mk_2op($2, $1, $3);
	}
	;

e8op:	'=' ;

e7:
	e6
	|
	e7 e7op e6
	{
	  $$ = mk_2op($2, $1, $3);
	}
	;

e7op:	LOGICAL_AND | LOGICAL_OR ;

e6:
	e5
	|
	e6 e6op e5
	{
	  $$ = mk_2op($2, $1, $3);
	}
	;

e6op:	'&' | '|' | '^' ;

e5:
	e4
	|
	e5 e5op e4
	{
	  $$ = mk_2op($2, $1, $3);
	}
	;

e5op:	'<' | '>' | EQUAL | NOT_EQUAL | GREATER_EQUAL | LESS_EQUAL ;

e4:
	e3
	|
	e4 e4op e3
	{
	  $$ = mk_2op($2, $1, $3);
	}
	;

e4op:	LSH | RSH ;

e3:
	e2
	|
	e3 e3op e2
	{
	  $$ = mk_2op($2, $1, $3);
	}
	;

e3op:   '+' | '-' ;

e2:
	e1
	|
	e2 e2op e1
	{
	  $$ = mk_2op($2, $1, $3);
	}
	;

e2op:   '*' | '/' | '%';

e1:
	e0
	|
	e1op e0
	{
	  $$ = mk_1op($1, $2);
	}
	;

e1op:	HIGH | LOW | '-' | '!' | '~';

e0:
	IDENTIFIER
        {
	  $$ = mk_symbol($1);
        }
	|
	NUMBER
	{
	  $$ = mk_constant($1);
	}
	|
        '$'
        {
	  $$ = mk_symbol("$");
	}
	|
	'(' expr ')'
	{
	  $$ = $2;
	}
	|
	'*' 
	{
	  $$ = mk_constant(TBL_NO_CHANGE);
	}
	| 
	TBL_POST_INC
	{
	  $$ = mk_constant($1);
	}
	| 
	TBL_POST_DEC
	{
	  $$ = mk_constant($1);
	}
	| 
	TBL_PRE_INC
	{
	  $$ = mk_constant($1);
	}
	;

%%
