/* Main function for gpasm
   Copyright (C) 1998 James Bowman

This file is part of gpasm.

gpasm is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

gpasm is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with gpasm; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include "stdhdr.h"

#include "gpasm.h"
#include "symbol.h"
#include "opcode.h"
#include "lst.h"
#include "hex.h"

struct gpasm_state state = {
    16,		/* radix */
    inhx8m,	/* hex_format */
    1,		/* case_insensitive */
    0		/* quiet */
};

char *include_paths[MAX_INCLUDE_PATHS];
int n_include_paths = 0;

int yyparse(void);

int main( int argc, char *argv[] )
{
  int c;
  int usage = 0;
  int memory_dump = 0;
  char *pc;

  i_memory_create();
  
  /* #defines are case-insensitive */
  state.stDefines = push_symbol_table(NULL, 1);
  state.pass = 0;

  while ((c = getopt(argc, argv, "?D:I:a:cd:e:hlmp:qr:w:v")) != EOF) {
    switch (c) {
    case '?':
    case 'h':
      usage = 1;
      break;
    case 'a':
      if (strcasecmp(optarg, "inhx8m") == 0)
	state.hex_format = inhx8m;
      else if (strcasecmp(optarg, "inhx8s") == 0)
	state.hex_format = inhx8s;
      else if (strcasecmp(optarg, "inhx32") == 0)
	state.hex_format = inhx32;
      else {
	fprintf(stderr,
		"Error: unrecognised hex file format \"%s\"\n",
		optarg);
      }
      break;
    case 'c':
      state.case_insensitive = 0;
      break;
    case 'D':
    case 'd':
      if ((optarg != NULL) && (strlen(optarg) > 0)) {
	struct symbol *sym;
	char *lhs, *rhs;

	lhs = strdup(optarg);
	rhs = strchr(lhs, '=');
	if (rhs != NULL) {
	  *rhs = '\0';	/* Terminate the left-hand side */
	  rhs++;	/* right-hand side begins after the '=' */
	}

	sym = get_symbol(state.stDefines, lhs);
	if (sym == NULL)
	  sym = add_symbol(state.stDefines, lhs);
	if (rhs)
	  annotate_symbol(sym, rhs);
      }
      break;
     case 'I':
       if(n_include_paths < MAX_INCLUDE_PATHS) {
 	 include_paths[n_include_paths++] = optarg;
       } else {
 	 fprintf(stderr, "too many -I paths\n");
 	 exit(1);
       }
       break;    
 
    case 'l':
      dump_processor_list();
      exit(0);
      break;
    case 'm':
      memory_dump = 1;
      break;
    case 'o':
	    strcpy(state.hexfilename, optarg);
	    strcpy(state.basefilename, optarg);
	    pc = strrchr(state.basefilename, '.');
	    if (pc)
		   *pc = 0;
      break;
    case 'p':
      select_processor(optarg);
      break;
    case 'q':
      state.quiet = 1;
      break;
    case 'r':
      select_radix(optarg);
      break;
    case 'v':
      fprintf(stderr, "%s\n", GPASM_VERSION_STRING);
      exit(0);
      
    }
    if (usage)
      break;
  }
  
  if (optind < argc)
    state.srcfilename = argv[optind];
  else
    usage = 1;

  if (usage) {
      printf("\n");
      printf("usage: gpasm <options> <filename>\n");
      printf("Where <options> are:\n");
      printf(" -a <fmt>       Select hex format inhx8m, inhx8s, inhx32\n");
      printf(" -c             case sensitive \n");
      printf(" -d <sym>=<val> Define symbol with value\n");
      printf(" -h             Show this usage message \n");
      printf(" -I             Specify include directory\n");
      printf(" -l             List supported processors\n");
      printf(" -m             Memory dump\n");
      printf(" -o <file>      Alternate name of hex output file\n");
      printf(" -p <processor> Select processor\n");
      printf(" -q             Quiet \n");
      printf(" -r <radix>     Select radix \n");
      printf(" -v             Show version \n");
      printf("\n");
      exit(0);
  }

  if(state.basefilename[0] == '\0') {
	  strcpy(state.basefilename, state.srcfilename);
	  pc = strrchr(state.basefilename, '.');
	  if (pc)
		  *pc = 0;
  }

  /* Builtins and macros are always case insensitive */
  state.stBuiltin = push_symbol_table(NULL, 1);
  state.stMacros = push_symbol_table(NULL, 1);
  state.stTop = 
    state.stGlobal = push_symbol_table(NULL, state.case_insensitive);

  opcode_init(0);

  state.maxram = (MAX_RAM - 1);

  open_src(SPECIAL_PATH, 0);

  state.pass = 1;

  yyparse();

  assert(state.pass == 2);
  
  pop_symbol_table(state.stBuiltin);
  
  dump_hex();

  if(memory_dump)
    print_i_memory();

  /* Maybe produce a symbol table */
  if (state.lst.symboltable) {
    lst_throw(); /* Start symbol table on a fresh page */
    dump_symbol_table(state.stGlobal);
  }

  /* Finish off the listing and symbol files*/
  lst_close();
  if (state.processor_info)
    sym_close();
  free_files();

  if (state.num.errors > 0)
    return 1;
  else
    return 0;
}
