/* i_memory.c Supports instruction memory.
   Copyright (C) 1998-2000 James Bowman, Scott Dattalo

This file is part of gpasm.

gpasm is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

gpasm is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with gpasm; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include "stdhdr.h"

#include "gpasm.h"
#include "i_memory.h"

/************************************************************************

 i_memory.c

    This file provides the functions used to manipulate the instruction
 memory. 
    The instruction memory is stored in 'memory blocks' which are implemented
 with the 'MemBlock' structure:

     typedef struct MemBlock {
       unsigned int base;
       unsigned int *memory;
       struct MemBlock *next;
     } MemBlock;

 Each MemBlock can hold up to `MAX_I_MEM' (64k currently) words. The `base'
 is the base address of the memory block. If the instruction memory spans
 more than 64k, then additional memory blocks can be allocated and linked
 together in a singly linked list (`next'). The last memory block in a 
 linked list of blocks has its next ptr set to NULL.
 
 ************************************************************************/

void i_memory_create(void)
{

  state.i_memory         = (MemBlock *)malloc(sizeof(MemBlock));
  state.i_memory->base   = 0;
  state.i_memory->memory = (unsigned int *)malloc(MAX_I_MEM * sizeof(unsigned int));
  state.i_memory->next   = NULL;

}

void i_memory_free(void)
{
  MemBlock *m,*n;
  m = state.i_memory;

  do {

    if(m->memory)
      free(m->memory);

    n = m->next;
    free(m);
    m = n;

  } while(m);
}

/************************************************************************
 * i_memory_new
 *
 *  Create memory for a new memory block.
 *
 *  Inputs:
 *   mpb  - pointer to the memory block structure (MemBlock)
 *   base_address - where this new block of memory is based
 *
 ************************************************************************/

MemBlock * i_memory_new(MemBlock *mbp, int base_address)
{
  MemBlock *m;
  int base;

  m = state.i_memory;
  base = base_address >> I_MEM_BITS;

  mbp->memory = (unsigned int *)malloc(MAX_I_MEM * sizeof(unsigned int));
  mbp->base   = base;

  do {

    if((m->next == NULL) || (m->next->base > base)) {
      // Insert after this block
      mbp->next = m->next;
      m->next   = mbp;
      return mbp;
    }

    m = m->next;
  } while(m);

  assert(0);
}

/************************************************************************
 * i_memory_get
 *
 * Fetch a word from the pic memory. This function will traverse through
 * the linked list of memory blocks searching for the address from the 
 * word will be fetched. If the address is not found, then `0' will be
 * returned.
 *
 * Inputs:
 *  address - 
 * Returns
 *  the word from that address
 *
 ************************************************************************/
int i_memory_get(unsigned int address)
{
  MemBlock *m = state.i_memory;

  do {
    assert(m->memory != NULL);

    if( (address >> I_MEM_BITS) == m->base )
      return m->memory[address & I_MEM_MASK];

    m = m->next;
  } while(m);

  return 0;  
}

/************************************************************************
 *  i_memory_put
 *
 * This function will write one word to a pic memory address. If the
 * destination memory block is non-existant, a new one will be created.
 *
 * inputs:
 *   address - destination address of the write
 *   value   - the value to be written at that address
 * returns:
 *   none
 *
 ************************************************************************/
void i_memory_put(unsigned int address, unsigned int value)
{
  MemBlock *m = NULL;

  do {

    if(m)
      m = m->next;
    else
      m = state.i_memory;

    if(m->memory == NULL) {
      i_memory_new(m,address);
    }

    if( (address >> I_MEM_BITS) == m->base ) {
      m->memory[address & I_MEM_MASK] = value;
      return;
    }
  } while (m->next);

  // Couldn't find an address to write this value. This must be
  // the first time we've tried to write to high memory some place.

  m = i_memory_new((MemBlock *) malloc(sizeof(MemBlock)), address);
  m->memory[address & I_MEM_MASK] = value;

}

/************************************************************************
 * 
 *
 *  These two functions are used to read and write instruction memory.
 * 
 *
 ************************************************************************/
void print_i_memory(void)
{
  MemBlock *m = state.i_memory;
  int base,i,j,row_used;
  char c;

#define WORDS_IN_ROW 8

  do {
    assert(m->memory != NULL);

    base = (m->base << I_MEM_BITS);

    for(i = 0; i<MAX_I_MEM; i+=WORDS_IN_ROW) {
      row_used = 0;

      for(j = 0; j<WORDS_IN_ROW; j++)
	if( m->memory[i+j] )
	  row_used = 1;

      if(row_used) {
	printf("%08X  ",(base+i) );

	for(j = 0; j<WORDS_IN_ROW; j++)
	  printf("%04X ", m->memory[i+j] & 0xffff );

	for(j = 0; j<WORDS_IN_ROW; j++) {
	  c = m->memory[i+j] & 0xff;
	  putchar( (isprint(c)) ? c : '.');

	  c = (m->memory[i+j]>>8) & 0xff;
	  putchar( (isprint(c)) ? c : '.');
	}
	putchar('\n');
      }
    }

    m = m->next;
  } while(m);

}

